------------------------------------------------------------------------------
-- Remote Monitor.lua -- Version 1.1
--
-- Utility script.
--
-- This script acts as a remote render monitor with Fusion 5.01
-- Updates: March 14 - fixed issue with setting timeout on connection. 
-- written by : sean konrad (sean@eyeonline.com)
-- updated    : Mar. 14, 2006
-- 
------------------------------------------------------------------------------


-- Notes: iup list box entries are not numerically indexed like lua tables -- instead, they are assigned
-- string based properties that are.. numbers.  So, for example, the first entry in the iup list box named iupList 
-- would be printed by typing the following: print(iupList["1"]).

-- Because the lists are generated dynamically and display the slave / job info, what I've done is keep track of the 
-- number of slaves/ jobs and then use for loops with the iterating value converted to a string.

-- This gives definition to the numbers returned by the 'status attributes
-- for render jobs.  This has been converted to a string variable from a numeric index.

-- To make this work properly, we needed a new function to check to make sure that the manager still existed as 
-- repeatedly connecting was doing, well, bad things.  As such, this script won't function in 5.0.  You probably
-- shouldn't use the release build anyway.

-- Beyond that, the vast majority of the iup interface was created by playing around until I got something that 
-- I felt was functional.  Debugging iup is extremely difficult, as it doesn't always point you to what line the 
-- script is failing on and may on occasion do things with Fusion that aren't exactly kosher.  Save your work if
-- you're playing around with the design and functionality of an iup script.
--ldofile(fusion:MapPath("Scripts:\\bmd.scriptlib"))

-- Create all of the buttons.  They will be positioned in the iup interface
-- later on in the code.
btn_refresh = iup.button{ title = "Refresh", size=80, FLAT="YES"}
btn_delete = iup.button{ title = "Delete Job", size = 80, FLAT="YES"}
btn_addjob = iup.button{ title = "Add Job", size = 80, FLAT="YES"}
btn_moveUp = iup.button{ title = "Move Job Up", size = 80, FLAT="YES"}
btn_moveDown = iup.button{ title = "Move Job Down", size = 80, FLAT="YES"}

btn_remSlave = iup.button{ title = "Remove Slave", size = 80, FLAT="YES"}
btn_addSlave = iup.button{ title = "Add Slave", size =80, FLAT="YES"}
btn_scan = iup.button{ title = "Scan for Slaves", size =80, FLAT="YES"}
textGroupsJob = iup.text{value = "", SIZE = "80",FGCOLOR = "200 200 200", MARGIN="15"}
textGroupsSlave = iup.text{value = "", SIZE = "80",FGCOLOR = "200 200 200", MARGIN="15"}
btn_GroupsJob = iup.button{ title = "Modify Groups", size =80, FLAT="YES"}
btn_GroupsSlave = iup.button{ title = "Modify Groups", size =80, FLAT="YES"}

btn_browse  = iup.button { title = "Browse", size =80, FLAT="YES", FGCOLOR = "200 200 200"}
textFileBox = iup.text{value = "c:\\program files\\fusion\\comps\\", SIZE = "200",FGCOLOR = "200 200 200", MARGIN="15"}
btn_OK = iup.button { title = "OK", size =80, FLAT="YES", FGCOLOR = "200 200 200", MARGIN = "47"}
btn_Cancel  = iup.button { title = "Cancel", size =80, FLAT="YES", FGCOLOR = "200 200 200", MARGIN ="47"}
btn_OKSlave = iup.button { title = "OK", size =80, FLAT="YES", FGCOLOR = "200 200 200", MARGIN = "47"}
btn_CancelSlave  = iup.button { title = "Cancel", size =80, FLAT="YES", FGCOLOR = "200 200 200", MARGIN ="47"}
textSlave = iup.text{value = "", SIZE = "200",FGCOLOR = "200 200 200", MARGIN="15"}

-- Connect to the local fusion interface.
if not fusion then
	-- The second argument is the timeout.  Otherwise it will try to connect forever.
	fusion = Fusion("localhost", 10)
end
-- Set the master variable which will be used to connect to the render manager in question.
-- If there's no instance of Fusion, set it to connect to localhost (which it won't be able to do).  Then let the user specify.
if not fusion then 
	master = "localhost" 
else 
	fusion:SetTimeout(0)
	master = fusion:GetPrefs()["Global"]["Network"]["ServerName"] 
end

-- Set up the connection button and its associated text box.  
btn_Connect = iup.button{ title = "Connect...", size = 80, FLAT="YES", MARGIN = "15"}
textConnect = iup.text{value = master, SIZE = "200",FGCOLOR = "200 200 200", MARGIN="15"}



function getfilename(path)
	for i = string.len(path), 1, -1 do
		teststring = string.sub(path, i, i)
		if teststring == "\\" or teststring == "/" then
			return string.sub(path, i+1)
		end
	end
end

-- This will format the slave's attributes so that it can be properly displayed later on in the interface.
-- Since this has been converted to the iup interface, the formatting hasn't been quite as perfect
-- as the console dump version of this script.

function fmtSlave(slave)
	
	-- Get the slave's attributes.
	slv = slave:GetAttrs()
	
	-- If it has priority classes?
	if slv.RSLVS_PriorityClasses == nil then classes = "N/A" else classes = slv.RSLVS_PriorityClasses end
	
	-- Get the slave's status.
	local status = slv.RSLVS_Status
	-- The string.format function is being used here to determine spacing for the 
	-- name, IP, version, status, and classes if they exist.  Return the string.
	
	return string.format("%-40.40s", slv.RSLVS_Name).." "..
	string.format("%-16.16s", slv.RSLVS_IP  ).." "..
	string.format("%-15.15s", slv.RSLVS_Version).." "..
	string.format("%-15.15s", tostring(status)).." "..
	string.format("%-10.10s", classes)
end

-- This will format the job's attributes so that it can be displayed in the iup list box.
-- This is having the same problem as the above function since its conversion to the iup format.

function fmtJob(job)
	a = job:GetAttrs()

	-- Get the total number of frames by adding the Unrendered, Rendered, and Rendering frames.
    total = a.RJOBN_UnrenderedFrames + a.RJOBN_RenderedFrames + a.RJOBN_RenderingFrames
	
	--Also get the total number of rendered frames.
    done  = a.RJOBN_RenderedFrames

	if a.RJOBS_PriorityClass == nil then
	   classes = nil
	else
	   classes = a.RJOBS_PriorityClass
	end
  
	if a.RJOBS_Type == "Comp" then
		-- Here we use string.format to define the spacing between job names.
		-- Incidentally, the only reason this works is because we're using a fixed width font
		-- in the list box (Lucida Console).  
		local strAssembly = string.format("%-35.35s", bmd.parseFilename(a.RJOBS_Name).FullName).." "..
		string.format("%-16.16s", a.RJOBS_QueuedBy).." "..
		string.format("%-15.15s", done.."/"..total).." "..
		string.format("%-15.15s", a.RJOBS_Status).." "
		if classes then
			strAssembly=strAssembly..string.format("%-10.10s", classes)
		end
		return strAssembly
	end
end

-- This function was added in to check if the host server is still actually connected.
-- If it isn't, error out.

function connectCheck()
	if masterConnect then 
		if masterConnect:IsAppConnected() == true then 
			return true
			
		end
	end
	errorConnect()
	return nil
end

-- Error if the host is no longer connected.

function errorConnect()
	
	clear_j()
	clear_s()

	j["1"] = "Could not connect to ".. master
end

-- The GetJobList() function can returns jobs that are in the process of being deleted
-- from the queue.  Unfortunately this sometimes takes longer than one would anticipate.  As such,
-- this function skips over jobs that are being removed (determined by the RJOBB_IsRemoving attribute)

function GetTrueJobList(renderMaster)
	jobList = {}
	jobListTemp = {}
	jobListTemp = renderMaster:GetJobList()
	for i, v in pairs(jobListTemp) do
		attrs = v:GetAttrs()
		if attrs.RJOBB_IsRemoving == false then
			
			-- only add to the slave list if it's not being removed.
			table.insert(jobList, v)
		end
	end
	
	-- return the slaveList variable.
	return jobList
end

-- The GetSlaveList() function occasionally returns slaves that are in the process of being deleted
-- from the queue.  Unfortunately this sometimes takes longer than one would anticipate.  As such,
-- this function skips over slaves that are being removed (determined by the RSLVB_IsRemoving attribute)

function GetTrueSlaveList(renderMaster)
	slaveList = {}
	slaveListTemp = {}
	slaveListTemp = renderMaster:GetSlaveList()
	for i, v in pairs(slaveListTemp) do
		attrs = v:GetAttrs()
		if attrs.RSLVB_IsRemoving == false then
			
			-- only add to the slave list if it's not being removed.
			table.insert(slaveList, v)
		end
	end
	
	-- return the slaveList variable.
	return slaveList
end

-- This will first clear and then refill both of the list boxes.
function fillList()
	clear_j()
	clear_s()
	fill_j()
	fill_s()
end

-- Clears the list box named "J"  
function clear_j()
	if jl then
		for i, v in pairs(jl) do
			j[tostring(i)]=nil
		end
	end
	-- If there's an error message in the list NOT part of the list box, clear it.
	if j["1"] then
		j["1"] = nil
	end
end

function fill_j()

	-- Fill the job list box.
	-- Check the connection.
	if connectCheck() then

		-- Get the job list to fill the list with...
		jl = nil
		jl = {}
		jl = GetTrueJobList(rm)
		for i, v in pairs(jl) do
			j[tostring(i)]=fmtJob(v)
		end
	end
end

function fill_s()
	if connectCheck() then
		-- Get the slave list to fill the list with...
		sl = nil
		sl = {}
		sl = GetTrueSlaveList(rm)
		for i, v in pairs(sl) do
			s[tostring(i)]=fmtSlave(v)
		end
	end
end


function clear_s()
	if sl then 
		for i, v in pairs(sl) do
			s[tostring(i)]=nil
		end
	end
end

-- This is the primary function used to connect to the master.  It's also used to create the list boxes if they haven't been created yet.

function connectMaster()

	-- a boolean is defined to keep track of whether or not the list boxes have been created yet.
	if listsCreated then
	
		-- if they've already been created, they obviously want to change which master they're connecting to.
		master = textConnect.value
		
		-- connect to an instance of fusion -- defined by the 'master' variable.  
		-- set the timeout to 5, and make sure we connect to a Render manager first.  
		-- if that fails, connect to the main instance of Fusion.
		
		-- we specify RenderManager, because there's a chance that it might connect to another 
		-- instance of Fusion first.  depending on which was opened first.
		masterConnect = Fusion(master, 5.0, nil, "RenderManager") 
		if masterConnect == nil then
			masterConnect = Fusion(master, 5.0) 
		end
		
		-- if it's connected display the lists.
		if connectCheck() == true then
			masterConnect:SetTimeout(0)
			-- also, define the rm variable.
			rm = masterConnect.RenderManager
			refreshMonitor()
		end
	else
	
		-- set the listsCreated boolean to true.
		listsCreated = true
		
		-- Define the size of the lists and create them using the iup.list() function...
		j = iup.list{SIZE="405x180"}
		s = iup.list{SIZE="405x100"}
		
		-- Deselect list box items.
		j.value = 0
		
		-- Set the BG color to something lighter than the main interface to give it contrast.
		j.BGCOLOR = "75 75 75"
		
		-- the font is lucide console because it a fixed width font.
		j.FONT = "lucida console::"
		
		-- Same for the slave list list box.
		s.value = 0
		s.BGCOLOR = "75 75 75"
		s.FONT = "lucida console::"
		
		-- Connect
		masterConnect = Fusion(master, 5.0, nil, "RenderManager") 
		if masterConnect == nil then
			masterConnect = Fusion(master, 5.0) 
		end
		if masterConnect then
			masterConnect:SetTimeout(0)
			rm = masterConnect.RenderManager
			fillList()
		else
			errorConnect()
		end
	end
end


-- Connect!! why not?
connectMaster()


----------------------------------------------------------------------------------------------------------------------------------------------------
-- This section is for spawning the interface of both the main window and the popup windows..-------------------------------------------------------
----------------------------------------------------------------------------------------------------------------------------------------------------

-- Using a vbox will make all items in it be aligned vertically.  If you're interested in further exposition about the interface
-- features of iup, it's suggested you read the actual documentation available on the iup website.  
vbox1 =   iup.vbox
  {
    iup.hbox
    {
               
        iup.vbox
        {
			-- buttons and groups text box go here.  FGCOLOR of the text is the off white colour.
			btn_refresh,
			btn_delete,
			btn_addjob,
			btn_moveUp,
			btn_moveDown,
			
			-- We define a label here, because its settings are minimal.
			iup.label{title = "Job's Groups:", FGCOLOR = "200 200 200"},
			textGroupsJob,
			btn_GroupsJob,
			
			SIZE = "180"
        },
        iup.vbox
        {
			iup.label{title = "Job List:", FGCOLOR = "200 200 200"},
			j
		},     
	-- Sets a 5 point gap between interface items.
     GAP = "5"
	},
	iup.hbox
    {
        iup.vbox
		{
			btn_addSlave,
			btn_remSlave,
			btn_scan,
			iup.label{title = "Slave's Groups:", FGCOLOR = "200 200 200"},
			textGroupsSlave,
			btn_GroupsSlave,
			SIZE = "100"
        },
                  
        iup.vbox
        {
			iup.label{title = "Slave List:", FGCOLOR = "200 200 200"},
			s
		}
	-- Also create a margin around the edge of the HBOX to keep it from touching the edge of the interface..
    ,GAP = 5, MARGIN = 5
	},
	iup.vbox{iup.hbox{}, GAP = "0",
	iup.hbox
		{ btn_Connect,
		textConnect
		,GAP = "15", SIZE = "180x90", ALIGNMENT = "ACENTER" 
		}},MARGIN = "5x5"}
  
  
vboxBrowse =   iup.vbox
  {
    iup.hbox
    {
      iup.frame({                   
        iup.hbox
        {
          btn_browse,
		  textFileBox,
		  SIZE = "270"
        },
        title="", FGCOLOR = "70 70 70", Size = "280x30"
      }),
     GAP = "", MARGIN = "10"
	},
	iup.vbox{iup.fill{SIZE = "5"}, SIZE = "5"},
	iup.vbox
    {
                     
        iup.hbox
        {
		btn_OK,
		btn_Cancel,
		 ALIGNMENT="ACENTER"
        },
        title="", SIZE = "180x30", ALIGNMENT = "ACENTER", FGCOLOR = "70 70 70", GAP = "50",
	ALIGNMENT = "ACENTER", MARGIN = "35"}
  }
  
vboxSlave =   iup.vbox
  {
    iup.hbox
    {
        iup.hbox
        {
		  iup.label{title = "HostID or IP:", FGCOLOR = "200 200 200"},
		  textSlave,
		  SIZE = "270", MARGIN = "10", GAP = "10"
        }
	  ,GAP = "", MARGIN = "10"
	},
	iup.vbox{iup.fill{SIZE = "5"},SIZE = "5"},
	iup.vbox
    {
                     
        iup.hbox
        {
		btn_OKSlave,
		btn_CancelSlave,
		 ALIGNMENT="ACENTER"
        },
        title="", SIZE = "180x30", ALIGNMENT = "ACENTER", FGCOLOR = "70 70 70", GAP = "50",
	ALIGNMENT = "ACENTER", MARGIN = "35"},
  MARGIN = "5x5"
  }
  
  
dlg = iup.dialog
{
  vbox1,
  title="IUP Fusion Render Monitor", menu=mnu, SIZE="680x300", BGCOLOR="60 60 60", FGCOLOR = "200 200 200", TOPMOST = "NO", FONT = "", RESIZE = "NO", MAXBOX = "NO"
}

----------------------------------------------------------------------------------------------------------------------------------------------------
-- This section is for functions related to file browsing for a .comp file to add to the manager.---------------------------------------------------
----------------------------------------------------------------------------------------------------------------------------------------------------

function addJob()
	dlg_filebrowse=iup.dialog
	{	
	vboxBrowse,
	title="Select your file...", menu=mnu, SIZE="315x55", BGCOLOR="70 70 70", FGCOLOR = "70 70 70", TOPMOST = "NO", FONT = "ARIAL ::", RESIZE = "NO", MAXBOX = "NO"
	}
end
function addJob_fileDLG()
	-- Creates a file dialog and sets its type, title, filter and filter info
	filedlg = iup.filedlg{dialogtype = "OPEN", title = "File Open", 
                      filter = "*.comp", filterinfo = "Fusion .comp File",
                      directory="c:\\program files\\fusion\\comp\\"} 

	-- Shows file dialog in the center of the screen
	filedlg:popup (iup.ANYWHERE, iup.ANYWHERE)
	-- Gets file dialog status
	status = filedlg.status
	-- if it found the file...

	
	return filedlg.value
	
end
----------------------------------------------------------------------------------------------------------------------------------------------------
-- This section is for functions related to file browsing for a .comp file to add to the manager.---------------------------------------------------
----------------------------------------------------------------------------------------------------------------------------------------------------

function refreshMonitor()
	
	-- The usual step for anything related to the manipulation of the queue is to check if it's conected...
	
	if connectCheck() then
		
		-- if it is, clear the boxes and refill them
		clear_j()
		fill_j()
		
		-- and the same for the slave boxes...
		clear_s()
		fill_s()
		
		-- clear the text boxes as nothing will be highlighted anymore.
		textGroupsJob.value = ""
		textGroupsSlave.value = ""
	end
end

----------------------------------------------------------------------------------------------------------------------------------------------------
------------- This area of the code exists primarily as an area for button press related functions -------------------------------------------------
----------------------------------------------------------------------------------------------------------------------------------------------------


function btn_refresh:action()
	-- Refill the list boxes.  The function refreshMonitor will check to make sre that it's still connected to Fusion
	-- for us, so no need to call it now.
	refreshMonitor()
	
  	return iup.DEFAULT
end


-- This is the action for the button that deletes the selected job from the queue.
function btn_delete:action()
	-- if the list even has an entry in it..
	if j["1"] then
	
		-- if something is selected...
		if j.value ~=0 then
		
			-- if it's connected ..
			if connectCheck() then 
				
				-- Get the job list to see if it still matches the old job list (in case the job list somehow got altered -- job already deleted, etc.).
				jl_new = nil
				jl_new = {}
				jl_new = GetTrueJobList(rm)
				
				-- Get the currently selected entry -- convert it to a number value so that it can actually be used 
				-- when we're iterating through tables..
				cur_Sel = tonumber(j.value)
				
				-- Compare the old list to the new..
				if tostring(jl[cur_Sel]) == tostring(jl_new[cur_Sel]) then
					
					-- Display a message in the monitor with using the Log() function..
					rm:Log("REMOTE IUP MONITOR job removed "..jl[cur_Sel]:GetAttrs().RJOBS_Name)
					
					-- Remove the job
					rm:RemoveJob(jl_new[cur_Sel])
					
					-- Give it a second to delete.
					wait(1)
					
					-- Refresh.
					refreshMonitor()
				else
				
					-- If the entries no longer match, display an error.
					clear_j()
					j["1"] = "Queue manipulated since last update -- please refresh and try again."
				end
			end
		end
	end
	return iup.DEFAULT
end


-- This is the button action for the removing of slaves..

function btn_remSlave:action()
	-- Do a similar set of checks to what we did above.
	if s["1"] then
		if s.value ~= 0 then
			if connectCheck() then
				sl_new = nil
				sl_new = {}
				
				-- The slave list that is returned by the function GetSlaveList() function
				-- sometimes has slaves that are being deleted, but are taking a second to disconnect
				-- therefore they don't show up in the slave list on the actual monitor.  We shouldn't display them either.
				-- Hence GetTrueSlaveList()
				sl_new = GetTrueSlaveList(rm)
				cur_Sel = tonumber(s.value)
				if tostring(sl[cur_Sel]) == tostring(sl_new[cur_Sel]) then
					rm:Log("REMOTE IUP MONITOR slave removed "..sl[cur_Sel]:GetAttrs().RSLVS_Name)
					rm:RemoveSlave(sl_new[cur_Sel]:GetAttrs().RSLVS_IP)
					
					wait(1)
					refreshMonitor()
				else
					clear_s()
					s["1"] = "Slave list modified since last update -- please refresh and try again."
				end
			end
		end
	end
	return iup.DEFAULT
end


-- This button scans for slaves.
function btn_scan:action()
	-- Check the connection.
	if connectCheck() then
	
		-- Scan for the slaves.  Render manager related function.
		rm:ScanForSlaves()
		
		-- Add a note to the log.
		rm:Log("REMOTE IUP MONITOR scanned for slaves.")
		wait(1)
		
		-- Refresh.
		refreshMonitor()
	end
	return iup.DEFAULT
end

-- The button that will move the job up in priority...
function btn_moveUp:action()

	-- Check the job list in a similar fashion to above..
	if j["1"] then
		if tonumber(j.value) ~= 0 then
			if connectCheck() then
				jl_new = nil
				jl_new = {}
				jl_new = GetTrueJobList(rm)
				cur_Sel = tonumber(j.value)
				if tostring(jl[cur_Sel]) == tostring(jl_new[cur_Sel]) then
					
					-- Print something in the log..
					rm:Log("REMOTE IUP MONITOR modified job priority "..jl[cur_Sel]:GetAttrs().RJOBS_Name.. " up.")
					
					-- Move the job up -- -1 moves it up one entry.
					rm:MoveJob(jl[cur_Sel] , -1)
					
					wait(1)
					refreshMonitor()
				else
					clear_j()
					j["1"] = "Queue modified since last update."
				end
			end
		end
	end
	return iup.DEFAULT
end

-- See above function.
function btn_moveDown:action()
	if j["1"] then
		if connectCheck() then
				if tonumber(j.value) ~= 0 then
				jl_new = nil
				jl_new = {}
				jl_new = GetTrueJobList(rm)
				cur_Sel = tonumber(j.value)
				if jl[cur_Sel] == jl_new[cur_Sel] then
					rm:Log("REMOTE IUP MONITOR move job priority "..jl[cur_Sel]:GetAttrs().RJOBS_Name.. " down.")
					
					-- A positive integer being passed into the second argument will move the priority down one.
					rm:MoveJob(jl[cur_Sel] , 1)
					wait(1)
					refreshMonitor()
				else
					clear_j()
					j["1"] = "Queue modified since last update."
				end
			end
		end
	end
	return iup.DEFAULT
end

-- This button (and associated text box) will modify the groups of the selected slave.
function btn_GroupsSlave:action()
	
	-- Yadda.
	if s["1"] then
		if s.value ~= 0 then
			sl_new = nil
			sl_new = {}
			
			if connectCheck() then
				sl_new = GetTrueSlaveList(rm)
				cur_Sel = tonumber(s.value)
				if tostring(sl[cur_Sel]) == tostring(sl_new[cur_Sel]) then
				
					-- Add something to the log.
					rm:Log("REMOTE IUP MONITOR modified Groups of slave : "..sl_new[cur_Sel]:GetAttrs().RSLVS_Name)
					
					-- Set RSLVS_Groups attribute equal to that of the text box.  Hopefully the user doesn't put any invalid slave
					-- groups into the text box.
					sl_new[cur_Sel]:SetAttrs({RSLVS_Groups = textGroupsSlave.value})
					
					wait(1)
					refreshMonitor()
				else
					-- Display an error.
					clear_s()
					s["1"] = "Slave list modified since last update -- please refresh and try again."
				end
			end
		end
	end
	return iup.DEFAULT
end

-- Holla back (see above).
function btn_GroupsJob:action()
	if j["1"] then
		if j.value ~=0 then
			if connectCheck() then
				jl_new = nil
				jl_new = {}
				jl_new = GetTrueJobList(rm)
				cur_Sel = tonumber(j.value)
				if tostring(jl[cur_Sel]) == tostring(jl_new[cur_Sel]) then
					rm:Log("REMOTE IUP MONITOR modified Groups of job : "..jl_new[cur_Sel]:GetAttrs().RJOBS_Name)
					jl_new[cur_Sel]:SetAttrs({RJOBS_Groups = textGroupsJob.value})
					
					wait(1)
					refreshMonitor()
				else
					clear_j()
					j["1"] = "Queue manipulated since last update -- please refresh and try again."
				end
			end
		end
	end
	return iup.DEFAULT
end
----------------------------------------------------------------------------------------------------------------------------------------------------
------------- This area of the code exists primarily as an area for list click related functions ---------------------------------------------------
----------------------------------------------------------------------------------------------------------------------------------------------------

-- When one clicks in the job list, it will change what's displayed in the groups text box. 
function j:action()
	if connectCheck() then		
		if j["1"] then
		
			-- if it's connected, then change the text box to equal the RJOBS_Groups attribute
			textGroupsJob.value = jl[tonumber(j.value)]:GetAttrs().RJOBS_Groups
		end
	end
end


-- See above.
function s:action()
	if connectCheck() then		
		if s["1"] then
			textGroupsSlave.value = sl[tonumber(s.value)]:GetAttrs().RSLVS_Groups
		end
	end
end
----------------------------------------------------------------------------------------------------------------------------------------------------
--------------- Functions related to the AddJob portion of the interface. --------------------------------------------------------------------------
----------------------------------------------------------------------------------------------------------------------------------------------------

-- This is the function that will run when the add job button is pressed.  It will
-- spawn an interface that will spawn a file browsing window when called upon.

function btn_addjob:action()
	
	-- Define the browse button action, which will spawn the file browsing window.
	function btn_browse:action()
	
		-- Set the fileLoc variable using the addjob_fileDLG() function.
		fileLoc = addJob_fileDLG()
		
		-- Set the text box equal to the value returned by fileLoc.  If fileLoc is nil, then it won't set the text
		-- box's value.
		if fileLoc then
			textFileBox.value = fileLoc
		end
	end
	
	-- Use the addjob function..
	addJob()
	
	-- show the dialog at the center of the screen.
	dlg_filebrowse:showxy(iup.CENTER, iup.CENTER)
	
	-- run this dialog's main loop.
	iup.MainLoop{}
end

-- The action for the OK button of the file browse dialog..
function btn_OK:action()
	
	-- Check to see if the file exists.
	if fileexists(textFileBox.value) == true then
		
		-- Add the job...
		rm:AddJob(textFileBox.value)
		
		-- Print in the log...
		rm:Log("REMOTE IUP MONITOR added job : "..textFileBox.value)
	end
	
	-- Reset the text box.
	textFileBox.value = ""
	
	-- Hide the interface.
	iup.Hide(dlg_filebrowse)
	
	-- Refresh.
	refreshMonitor()
end

function btn_Cancel:action()
	
	-- Clear the text box
	textFileBox.value = ""
	
	-- Hide the interface.
	iup.Hide(dlg_filebrowse)
	
	-- Refresh for good measure.
	refreshMonitor()
end

----------------------------------------------------------------------------------------------------------------------------------------------------
--------------- Functions related to the AddSlave portion of the interface. ------------------------------------------------------------------------
----------------------------------------------------------------------------------------------------------------------------------------------------
function addSlave()
	
	-- We need to show the dialog to add the slave -- we've already defined the elements for it above.
	dlg_slaveadd=iup.dialog
	{	
	vboxSlave
	,title="Enter the Slave's name...", menu=mnu, SIZE="315x55", BGCOLOR="70 70 70", FGCOLOR = "70 70 70", TOPMOST = "NO", FONT = "ARIAL::", RESIZE = "NO", MAXBOX = "NO"
	}
end


-- The action for the btn_addSlave button..
function btn_addSlave:action()
	-- Calls the already declared "addSlave() function 
	addSlave()
	
	-- Shows the dialog that's been created for the slave add..
	dlg_slaveadd:showxy(iup.CENTER, iup.CENTER)
	iup.MainLoop{}
end

-- Action for the OK button..
function btn_OKSlave:action()
	if connectCheck() then
	-- Check the connection, as per usual.
	-- Add the slave.
		rm:AddSlave(textSlave.value)
		
		-- We don't need to do any special to check to see if the slave exists or not
		-- at this point, becuase Fusion network rendering is generally smart enough to not add nonexistant slaves.  
	
		-- or it might add them, but it won't do anything with them.
		rm:Log("REMOTE IUP MONITOR added slave : "..textSlave.value)
	end
	
	textSlave.value = ""
	
	-- Hide the interface.
	iup.Hide(dlg_slaveadd)
	
	-- Check the connection! Everybody! 
	if connectCheck() then

		-- Refresh the monitor.
		refreshMonitor()
	end
end

-- Button for the cancel option..
function btn_CancelSlave:action()
	textSlave.value = ""
	
	-- Hide the dialog.
	iup.Hide(dlg_slaveadd)
	
	-- REFRESH
	refreshMonitor()
end


----------------------------------------------------------------------------------------------------------------------------------------------------
--------------- Connect button. --------------------------------------------------------------------------------------------------------------------
----------------------------------------------------------------------------------------------------------------------------------------------------


function btn_Connect:action()
	connectMaster()
end



----------------------------------------------------------------------------------------------------------------------------------------------------
--------------- Automatic Refresh. -----------------------------------------------------------------------------------------------------------------
----------------------------------------------------------------------------------------------------------------------------------------------------

-- One of the nice features of IUP is the ability to create a timer
-- that will call an action every X units of time (milliseconds).
timer = iup.timer{time=30000.0, run="YES"}

 function timer:action_cb()
	if connectCheck() then
		refreshMonitor()
		return iup.DEFAULT
	else
		connectMaster()
	end
 end 


----------------------------------------------------------------------------------------------------------------------------------------------------
-- Script host code starts here --
-- New to version 5.01 -- Now a person would in theory be able to script interfaces created by iup through another SEPARATE copy of fuscript.  
-- In this situation it probably isn't extremely useful.  It mainly exists as an example of how one would use this sort of thing.
-- To script something like this:

-- > mon = ScriptApp("RemoteMonitor", "localhost")
-- > =mon
-- RemoteMonitor (0x00c9de10) [App: 'RemoteMonitor' on 127.0.0.1, UUID: 9bd39c11-218b-416c-a254-cac81d641e27]
-- > mon:Refresh()
-- > =mon.Master
-- localhost
-- > mon.Master = "thrillhouse"
-- > =mon.Master
-- thrillhouse
-- > mon:Quit()
----------------------------------------------------------------------------------------------------------------------------------------------------

--~ Object = host.Class("Object", nil, 
--~ {
--~ 	Constructor = function(self, obj)
--~ 	end,
--~ })

--~ Application = host.Class("RemoteMonitor", Object,
--~ {
--~ 	Functions =
--~ 	{
--~ 		Quit = { "", "", function(self) host.ExitLoop(), end },
--~ 		Refresh = { "", "", function(self) refreshMonitor(), end },
--~ 	},
--~ 	
--~ 	Variables =
--~ 	{
--~ 		Master = { "", "",
--~ 			function(self, set, val)
--~ 				if set then
--~ 					textConnect.value = val
--~ 					masterConnect()
--~ 				else
--~ 					return textConnect.value
--~ 				end
--~ 			end
--~ 			},
--~ 	},
--~ 	
--~ 	Constants =
--~ 	{
--~ 	},
--~ 	
--~ 	Methods =
--~ 	{
--~ 	},
--~ 	
--~ 	Constructor = function(self, obj, blah)
--~ 		self.BaseClass:Constructor(obj)
--~ 	end,
--~ })

--~ app = Application()

--~ host.SetHostApp(app, "RemoteMonitor", {})

-------------------------------------------
------------------------------------------

-- Show the interface
dlg:showxy(iup.CENTER, iup.CENTER)
-- Throw it into the main loop.
iup.MainLoop()
